package gov.cms.grouper.snf.app;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.spy;

import gov.cms.grouper.snf.lego.ActionWrapper;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.lego.SupplierWithException;
import gov.cms.grouper.snf.lego.Triple;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.transfer.SnfClaim;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.junit.Assert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;

public class PdpmTest {

  private static final Function<Integer, Integer> increment = (counter) -> ++counter;

  private static final String numOfCorrectKey = "numOfCorrect";
  private static final String misMatchHippsCounterKey = "misMatchHippsCount";
  public static final String totalNumberRecordKey = "numberOfRecords";
  public static final String errorCounterKey = "errorCount";

  public static final Logger logger = LogManager.getLogger();

  public static void main(String[] args) throws Throwable {
    final boolean dryRun = "true".equals(System.getProperty("dryRun"));

    LoggerContext ctx = (LoggerContext) LogManager.getContext(false);
    Configuration config = ctx.getConfiguration();
    LoggerConfig loggerConfig = config.getLoggerConfig(LogManager.ROOT_LOGGER_NAME);
    String logLevel = System.getProperty("snfLog");
    if (logLevel != null) {
      System.out.println("*******************");
      logger.debug("log4j " + logLevel + " On");
      loggerConfig.setLevel(Level.valueOf(logLevel));
    } else {
      loggerConfig.setLevel(Level.OFF);
    }
    ctx.updateLoggers();


    String outputPath = "test_results/";
    String inputPath = "test_data";
    Integer initVersion = null;

    if (args.length > 0) {
      int ver = Integer.parseInt(args[0]);
      if (ver >= 0) {
        initVersion = ver;
      }
    }
    final Integer version = initVersion;

    if (args.length > 1) {
      inputPath = args[1];
    }

    // File testDir = new File(inputDir);
    Path inputDir = Paths.get(inputPath);
    Path outputDir = Paths.get(outputPath);
    if (dryRun == false) {
      outputDir.toFile().mkdirs();
    }

    System.out.println(
        "*** running with version: " + version + " testDataDirectory: " + inputDir.getFileName());

    File files[] = inputDir.toFile().listFiles(
        (pathname) -> pathname.isDirectory() == false && pathname.getName().endsWith(".txt"));
    for (File file : files) {
      System.out.println("**** Processing File: " + file.getName());
      List<Triple<Integer, String, Function<Integer, Integer>>> counterEntries = Arrays.asList(
          Triple.of(0, numOfCorrectKey, increment),
          Triple.of(0, misMatchHippsCounterKey, increment),
          Triple.of(0, totalNumberRecordKey, increment), Triple.of(0, errorCounterKey, increment));


      ActionWrapper<String, Integer> counters = new ActionWrapper<>(counterEntries);

      SupplierWithException<Writer> ws = PdpmTest.getWriter(dryRun, inputDir, file, outputDir);
      Function<Writer, Consumer<SnfClaim>> getProcessing = qaProcessing(dryRun, counters);
      Consumer<Writer> exec = PdpmTest.getExec(version, file, getProcessing, counters);

      PdpmTest.doMain(version, file, ws, exec);
      System.out.println("***********************************");
    }

  }

  protected static void doMain(Integer version, File inputFile, SupplierWithException<Writer> ws,
      Consumer<Writer> exec) throws Throwable {
    try (Writer writer = ws.get()) {
      exec.accept(writer);
    } catch (Throwable ex) {
      System.out.println("ERROR WHILE PROCESSING: " + inputFile.getAbsolutePath());
      throw new RuntimeException(ex);
    }

  }

  protected static Consumer<Writer> getExec(Integer version, File inputFile,
      Function<Writer, Consumer<SnfClaim>> getProcessing, ActionWrapper<String, Integer> counters) {
    Consumer<Writer> exec = (writer) -> {
      Consumer<SnfClaim> processing = getProcessing.apply(writer);
      List<String> errors = new ArrayList<>();
      Pdpm pdpm = new Pdpm(errors);

      timeProcess(() -> {
        pdpm.exec(version, inputFile.toPath(), processing);
        return counters;
      });
    };

    return exec;
  }

  protected static SupplierWithException<Writer> getWriter(boolean dryRun, Path inputDir,
      File inputFile, Path resultDir) {

    SupplierWithException<Writer> ws = () -> new Writer() {
      @Override
      public void write(String str) throws IOException {

      }

      @Override
      public void close() throws IOException {

      }

      @Override
      public void write(char[] cbuf, int off, int len) throws IOException {}

      @Override
      public void flush() throws IOException {
        // TODO Auto-generated method stub

      }
    };

    if (dryRun == false) {
      String inputRelDir = inputFile.getParentFile().getAbsolutePath()
          .replace(inputDir.toFile().getAbsolutePath(), "");
      String resultFilePath = resultDir.toFile().getAbsolutePath() + inputRelDir;
      File resultFileDir = new File(resultFilePath);
      resultFileDir.mkdirs();
      File resultFile = new File(resultFileDir.getAbsolutePath() + "/" + inputFile.getName());
      ws = () -> {
        return new BufferedWriter(new FileWriter(resultFile));
      };
    }

    return ws;
  }



  protected static void timeProcess(Supplier<ActionWrapper<String, Integer>> exec) {

    long begin = System.currentTimeMillis();
    ActionWrapper<String, Integer> counters = exec.get();
    long end = System.currentTimeMillis();

    Integer totalNumberOfRecords = counters.getValue(totalNumberRecordKey);
    Integer errorCount = counters.getValue(errorCounterKey);
    Integer numberOfCorrect = counters.getValue(numOfCorrectKey);
    Integer misMatchHippsCount = counters.getValue(misMatchHippsCounterKey);

    performanceTime(begin, end, totalNumberOfRecords, errorCount, misMatchHippsCount,
        numberOfCorrect);


  }

  /**
   * Formats a field from the MDS form and substitutes the formatted value in the fixed length
   * string.
   *
   * @param sb the string builder for the claim in fixed length format
   * @param field the field from the MDS form to be updated
   * @param rightJustified whether the field should be right justified (true) or left justified
   *        (false)
   * @param value the value of the field to be formatted
   */
  protected static void formatFixedLengthValue(StringBuilder sb, Rai300 field,
      boolean rightJustified, String value) {
    String justification = rightJustified ? "" : "-";
    sb.replace(field.getIndex() - 1, field.getIndex() + field.getLength() - 1,
        String.format("%1$" + justification + field.getLength() + "s", value));
  }


  private static void performanceTime(long begin, long end, int totalNumberOfRecords,
      int errorCount, Integer misMatchHippsCount, Integer numberOfCorrect) {
    final long processTime = end - begin;
    final long min = processTime / (1000 * 60);
    final long sec = (processTime % (((min == 0) ? 1 : min) * 1000 * 60)) / 1000;
    final long totalSec = processTime / 1000;

    if (numberOfCorrect != null) {
      double percentage = numberOfCorrect * 100.00 / totalNumberOfRecords;
      System.out.println("\t\tRecords Processed: " + totalNumberOfRecords);
      final int errors = (errorCount + misMatchHippsCount);
      if (errors != 0) {
        final double errorPercent = (100 - percentage);
        System.out.println("\t\terrors: " + errors + " (" + errorPercent + "%)");
      }
    } else {
      System.out.println("\t\tRecords Processed: " + totalNumberOfRecords);
      System.out.println("\t\terrors: " + errorCount);
    }

    String intervalFmt = "\t\ttotal process time with %d records = %d min and %d sec (%d Seconds)";
    System.out.println(String.format(intervalFmt, totalNumberOfRecords, min, sec, totalSec));

    long recordPerSecond = totalNumberOfRecords;
    if (totalSec > 0) {
      recordPerSecond = totalNumberOfRecords / totalSec;
    }
    System.out.println(String.format("\t\trecords per second = %d", recordPerSecond));

  }

  public static void printUsage() {
    System.out.println(
        "to use applicable associated assessment date version in the record: java -jar snf-1.1.jar -1 My/File/Directory my/Result/Dir");
    System.out.println(
        "to turn on log java -cp snf-{snfVersion}.jar -DsnfLog=ON gov.cms.grouper.snf.app.Pdpm {version} My/File/Directory my/Result/Dir");
    System.out.println(
        "please provide test files directory as args, example: java -jar snf-1.1.jar {version} My/File/Directory my/Result/Dir");
  }



  protected static Consumer<Writer> getExec(boolean dryRun, Integer version, File inputFile,
      Function<Writer, Consumer<SnfClaim>> getProcessing, ActionWrapper<String, Integer> counters) {

    Consumer<Writer> exec = (writer) -> {
      Consumer<SnfClaim> processing = getProcessing.apply(writer);
      List<String> errors = new ArrayList<>();
      Pdpm pdpm = new Pdpm(errors);

      if (dryRun) {
        Function<String, String> passThur = (line) -> line;
        List<String> lines = pdpm.processFile(version, inputFile.toPath(), passThur);
        PdpmTest.timeProcess(() -> {
          pdpm.exec(version, lines, processing);
          return counters;
        });
      } else {
        PdpmTest.timeProcess(() -> {
          pdpm.exec(version, inputFile.toPath(), processing);
          return counters;
        });
      }

    };

    return exec;


  }

  protected static Function<Writer, Consumer<SnfClaim>> qaProcessing(boolean dryRun,
      ActionWrapper<String, Integer> counters) throws Throwable {
    Function<Writer, Consumer<SnfClaim>> r = (writer) -> (claim) -> {
      counters.doAction(totalNumberRecordKey);
      StringBuilder sb = new StringBuilder(claim.getOriginalRecord());


      final String expected = getOriginalRecordHippsCode(claim);
      final String actual = claim.getHippsCode();

      if (claim.hasError()) {
        counters.doAction(errorCounterKey);
        PdpmTest.formatFixedLengthValue(sb, Rai300.ERROR_REASON, false,
            claim.getErrors().toString());
        PdpmTest.formatFixedLengthValue(sb, Rai300.RECALCULATED_Z0100A, false,
            claim.getHippsCode());
        sb.append(System.lineSeparator());
      } else if (expected.equals(actual) == false) {
        PdpmTest.formatFixedLengthValue(sb, Rai300.RECALCULATED_Z0100A, false,
            claim.getHippsCode());
        counters.doAction(misMatchHippsCounterKey);
        PdpmTest.formatFixedLengthValue(sb, Rai300.ERROR_REASON, false,
            "Z0100A & Recalculated_Z0100A mismatched.");
        sb.append(System.lineSeparator());
      } else {
        PdpmTest.formatFixedLengthValue(sb, Rai300.RECALCULATED_Z0100A, false,
            claim.getHippsCode());
        counters.doAction(numOfCorrectKey);
        sb.append(System.lineSeparator());
      }


      if (dryRun == false) {
        SnfUtils.doOrDie(() -> writer.write(sb.toString()));
      }
    };

    return r;
  }



  private static String getOriginalRecordHippsCode(SnfClaim claim) {
    int start = Rai300.Z0100A.getIndex() - 1;
    int end = start + Rai300.Z0100A.getLength() - 1;
    return claim.getOriginalRecord().substring(start, end).trim();
  }


  /*
   * @Test public void processTest() { Path path = SnfUtils.doOrDie(() ->
   * Paths.get(ClassLoader.getSystemResource("Nurse test.txt").toURI())); Pdpm pdpm = new
   * Pdpm(path);
   *
   * List<String> lines = pdpm.readAllLines(path); pdpm.setSnfClaims(pdpm.transformToClaims(lines));
   * pdpm.process(100, pdpm.getSnfClaims());
   *
   * Assertions.assertEquals("KIAA1", pdpm.getSnfClaims().get(0).getHippsCode());
   *
   * try { List<SnfClaim> claims = pdpm.transformToClaims(Arrays.asList("line that fail"));
   * pdpm.process(100, claims); Assertions.fail(); } catch (Throwable ex) {
   * Assertions.assertTrue(true); } }
   */

  @Test
  public void readAllLinesToClaimsTest() throws Exception {
    Pdpm pdpm = new Pdpm();

    String line = "MDS       NC 1.17      3.00      PIA765045          521230861CMS              "
        + "             help@qtso.com                                     jRAVEN                 "
        + "                           1.1.9               ^                   ^         765045   "
        + "   ^              10101  0103FPDPMf150   ^GrprNRSEf150      ^  9990211505GR1PR1NR50 ^ "
        + "             1193205010000010^              4^           ^         ^                  "
        + "    ^                      0100000201910011012019100601^       201910060^       ^     "
        + "  0010000112121222120^^^^^^^^     111101111111111111108 ^^^^^^^^^^^^^^^^^^^^^  "
        + "0010000^^^^^^00^^31111212232332221110"
        + "^^^^^^^^^^^^^^^^^^^^^11110000000000000000000000000000010000010"
        + "---0000011000000000000001000000000000000100000010000010000000000^       ^       ^     "
        + "  ^       ^       ^       ^       ^       ^       ^       00010^^^^ "
        + "^^^^^^^^100000000011110^^^00010671272       00000001000100^^^        ^^^^^^^^^^       "
        + "             0000000100000000010^^        0000000001010001000000 0010120191001^1^200 -"
        + "   -   52019100120191005200 -   -   52019100120191005200 -   -   5201910012019100515  "
        + "115  115  17700006000- - 00000000109^^1  ^2049920191001- - - "
        + "0-1-0-0-1-1-101-0-1010100-100-10100-0-0-20191006201910061^^           ^               "
        + "  ^^       ^        ^ ^    ^       ^       ^       ^ ^^^^^^^^           ^             "
        + "    ^       KIAA1  1.0003                      RAA       1.0448    RAA       1.0448   "
        + " 20191006^           1^^^01100 1 000001234567^^^ 32 12-   -   -   520191001011000401^ "
        + "0501^ 0401^ 0603^ 0605^ 0505^ 0506^ 0502^   0603^ 0605^ 1^0404^ 1^0505^ "
        + "1^^0112019100512019100511111132181000000605^ 0302^ 0709^ 88- ^ 0605^ 0106^ 0403^ 0403^"
        + " 0403^ 0403^ 88- ^ 0403^ 13        111^0^1^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ A20"
        + ".7^^^111111111111111111111111111111^   ^   ^   ^   ^  ^   ^   ^   ^   ^  ^   ^   ^   ^"
        + "   ^  ^    ^                                                                          "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                      KIAA1                           "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                   %";
    SnfClaim claim = pdpm.transformToClaim(line);
    Assertions.assertEquals("A20.7^^^", claim.getPrimaryDiagnosis().getValue());
    Assertions.assertEquals(1, claim.getAiCode().intValue());
    Assertions.assertEquals(1, claim.getObra().intValue());

    ClaimInfo claimUtil = new ClaimInfo(100, false, claim.getAssessments());

    Assertions.assertEquals(7, claimUtil.getAssessmentValue(Rai300.O0500A));
  }

  @Test
  public void getVersionTest() {
    Pdpm pdpm = new Pdpm();
    Assertions.assertEquals(104, pdpm.getVersion(LocalDate.of(2020, 4, 1)));
    Assertions.assertEquals(100, pdpm.getVersion(LocalDate.of(2019, 10, 1)));
    Assertions.assertNull(pdpm.getVersion(LocalDate.of(2019, 9, 30)));
  }

  @Test
  public void execLineTest() {
    String line = "          IPA1.17      3.00      PIA1429074         521230861CMS              "
        + "             help@qtso.com                                     jRAVEN                 "
        + "                           1.1.9               ^                   ^         760067   "
        + "   ^              29908  0993IPA1Kv2-655 ^LNPDPMv2-655      ^  999221655RRF999221655^ "
        + "             21932    0000011FRENCH         3^           ^         ^                  "
        + "    ^                                                          201910010^       ^     "
        + "  0   0   13221221130^      ^     11010101010101010133  ^^^^^^^^^^^^^^^^^^^^^  001000 "
        + "      00                                                                              "
        + "   001  -  0            0    000   00         000 0 00 0  00      10 0M12.851^K85"
        + ".90^^I10.^^^^E11.319^Z85.42^^I25.10^^N39.0^^^F32.89^^E10.319^^                        "
        + "   01  00  1       00001     0                    1 0         1 -   0                 "
        + "      000000011100000100 40         0 0 0 0 0 0   0 0 0    0 1                        "
        + "                                                                               0      "
        + "    0000000000                                                                        "
        + "                        1^^           ^                 ^^       ^        ^ ^    ^    "
        + "                   ^ ^^^^^^^^           ^                 ^       GBKD0  1.0003       "
        + "                                                       20191007^           ^    0000 1"
        + "  10             ^^ ^^                                                                "
        + "                                                                                      "
        + "                                                                 13            0  "
        + "0502050388040103048805S82.452D100000000000000000000000000001                          "
        + "                                    ^                                                 "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                               GBKE0  "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                                      "
        + "                                                                Z0100A & "
        + "Recalculated_Z0100A mismatched.                                                       "
        + "                                                                                      "
        + "                   %";

    Pdpm pdpm = new Pdpm(null);
    SnfClaim claim = pdpm.transformToClaim(line);
    SnfClaim finishedClaim = pdpm.process(null, claim);

    Assert.assertEquals(getOriginalRecordHippsCode(finishedClaim), finishedClaim.getHippsCode());
  }


  @Test
  public void execU071Test() throws Exception {
    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("U071 test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    Consumer<SnfClaim> test = (claim) -> {
      Assert.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(104, path, test);

    List<SnfClaim> results = pdpm.exec(104, path);
    for (SnfClaim claim : results) {
      Assert.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    }

    List<String> lines = pdpm.processFile(104, path, (line) -> line);
    results = pdpm.exec(104, lines);
    for (SnfClaim claim : results) {
      Assert.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    }

    SnfClaim claim = pdpm.transformToClaim(lines.get(0));
    claim = pdpm.execClaim(104, claim);
    Assert.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());

    claim = pdpm.exec(104, lines.get(0));
    Assert.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());

    pdpm = spy(pdpm);
    doReturn(null).when(pdpm).getVersion(any());
    claim = spy(claim);
    doThrow(RuntimeException.class).when(claim).getPrimaryDiagnosis();
    doThrow(NullPointerException.class).when(claim).addCode(any());

    // TODO confirm there should be error here or not
    // SnfClaim c = pdpm.execClaim(null, claim);
    // Assert.assertTrue(c.hasError());
  }

  @Test
  public void tryExecTest() {
    Pdpm pdpm = new Pdpm(null);
    try {
      pdpm.tryExec("", () -> {
        throw new Exception("error");
      });
    } catch (Exception ex) {
      Assert.assertTrue(true);
    }

  }


  @Test
  public void execNTATest() throws Exception {
    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("NTA test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    Consumer<SnfClaim> test = (claim) -> {
      Assertions.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(null, path, test);
  }

  @Test
  public void execNurseTest() throws Exception {
    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("Nurse test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    Consumer<SnfClaim> test = (claim) -> {
      Assertions.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(100, path, test);
  }

  @Test
  public void execPTOTTest() throws Exception {
    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("PTOT test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    Consumer<SnfClaim> test = (claim) -> {
      Assertions.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(100, path, test);

  }

  @Test
  public void execSLPTest() throws Exception {
    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("SLP test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    Consumer<SnfClaim> test = (claim) -> {
      Assertions.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(100, path, test);

    pdpm.exec(100, path);

  }


  @Test
  public void getClaimFixedStringTest() throws Exception {

    Path path =
        SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("Nurse test.txt").toURI()));
    Pdpm pdpm = new Pdpm();

    AtomicReference<SnfClaim> first = new AtomicReference<>();

    Consumer<SnfClaim> test = (claim) -> {
      if (first.get() == null) {
        first.set(claim);
      }
      Assertions.assertEquals(getOriginalRecordHippsCode(claim), claim.getHippsCode());
    };

    pdpm.exec(100, path, test);

    String hipps = "AFAA1  ";

    SnfClaim claim = first.get();
    claim.setHippsCode(hipps);

    StringBuilder sb = new StringBuilder(claim.getOriginalRecord());
    PdpmTest.formatFixedLengthValue(sb, Rai300.RECALCULATED_Z0100A, false, claim.getHippsCode());

    Assertions.assertEquals(sb.toString(), pdpm.getClaimFixedString(claim));
  }


  @Test
  @Order(0)
  public void testMain() throws Throwable {
    System.setProperty("dryRun", "true");

    // Assert.assertTrue(absolutePath.endsWith("src/test/resources/mainTest"));
    Path dir = SnfUtils.doOrDie(() -> Paths.get(ClassLoader.getSystemResource("mainTest").toURI()));
    dir.toFile().list();

    String testFileDir = dir.toString();
    String[] args = new String[] {"-1", testFileDir};

    boolean s = false;
    try {
      PdpmTest.main(args);
      args = new String[] {"100", testFileDir};
      main(args);
      s = true;
      Assertions.assertTrue(s);
    } catch (Exception ex) {
      Assertions.fail();
    }

    s = false;
    try {
      args = new String[] {"-1", testFileDir, "blah"};
      main(args);
      s = true;
      Assertions.assertTrue(s);
    } catch (Exception ex) {
      Assertions.fail();
    }


  }
}
